<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\LoanOffer;
use Illuminate\Support\Facades\DB;

class UpdateTakeHomeAmounts extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'loans:update-take-home-amounts';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Update take_home_amount for all existing loan offers based on requested amount minus charges';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting to update take_home_amount for all loan offers...');
        
        // Get all loan offers that need updating
        $loanOffers = LoanOffer::whereNull('take_home_amount')
            ->orWhere('take_home_amount', 0)
            ->get();
        
        $totalCount = $loanOffers->count();
        $updatedCount = 0;
        $failedCount = 0;
        
        if ($totalCount == 0) {
            $this->info('No loan offers need updating.');
            return Command::SUCCESS;
        }
        
        $this->info("Found {$totalCount} loan offers to update.");
        $bar = $this->output->createProgressBar($totalCount);
        $bar->start();
        
        foreach ($loanOffers as $loanOffer) {
            try {
                // Calculate take_home_amount based on the formula from the calculator
                // Take home amount = Principal (requested_amount) - Total Charges
                
                $requestedAmount = $loanOffer->requested_amount ?? 0;
                $processingFee = $loanOffer->processing_fee ?? 0;
                $insurance = $loanOffer->insurance ?? 0;
                $otherCharges = $loanOffer->other_charges ?? 0;
                
                // Calculate total charges
                $totalCharges = $processingFee + $insurance + $otherCharges;
                
                // Calculate take home amount (what customer actually receives)
                $takeHomeAmount = $requestedAmount - $totalCharges;
                
                // Update the record
                $loanOffer->take_home_amount = $takeHomeAmount;
                $loanOffer->save();
                
                $updatedCount++;
                
                // Log details for tracking
                $this->line('');
                $this->info("Updated Loan #{$loanOffer->id} - Check Number: {$loanOffer->check_number}");
                $this->line("  Requested Amount: " . number_format($requestedAmount, 2));
                $this->line("  Processing Fee: " . number_format($processingFee, 2));
                $this->line("  Insurance: " . number_format($insurance, 2));
                $this->line("  Other Charges: " . number_format($otherCharges, 2));
                $this->line("  Total Charges: " . number_format($totalCharges, 2));
                $this->line("  Take Home Amount: " . number_format($takeHomeAmount, 2));
                
            } catch (\Exception $e) {
                $failedCount++;
                $this->error("Failed to update Loan #{$loanOffer->id}: " . $e->getMessage());
            }
            
            $bar->advance();
        }
        
        $bar->finish();
        $this->line('');
        $this->info('Update complete!');
        $this->info("Successfully updated: {$updatedCount} loan offers");
        
        if ($failedCount > 0) {
            $this->warn("Failed to update: {$failedCount} loan offers");
        }
        
        // Show summary statistics
        $this->line('');
        $this->info('Summary Statistics:');
        
        $stats = DB::table('loan_offers')
            ->selectRaw('
                COUNT(*) as total_records,
                COUNT(take_home_amount) as records_with_take_home,
                AVG(requested_amount) as avg_requested,
                AVG(take_home_amount) as avg_take_home,
                AVG(requested_amount - take_home_amount) as avg_charges
            ')
            ->first();
        
        $this->table(
            ['Metric', 'Value'],
            [
                ['Total Records', number_format($stats->total_records)],
                ['Records with Take Home Amount', number_format($stats->records_with_take_home)],
                ['Average Requested Amount', 'TZS ' . number_format($stats->avg_requested, 2)],
                ['Average Take Home Amount', 'TZS ' . number_format($stats->avg_take_home, 2)],
                ['Average Total Charges', 'TZS ' . number_format($stats->avg_charges, 2)],
            ]
        );
        
        return Command::SUCCESS;
    }
}